//
//  BITaxManager.h
//  BISubstrate
//
//  Created by Eric Hochmeister on 01/09/06.
//  Copyright 2006 __MyCompanyName__. All rights reserved.
//

/*!
    @class       BITaxManager
    @abstract    Manages taxes (versioning, adding taxes, deleting taxes, etc.)
    @discussion  (comprehensive description)
*/


#import <Cocoa/Cocoa.h>

@class MCPObjectContext;
@class BITax;
@class BIConsolidatedTax;
@class BITaxConsolidatedTaxEntry;

@interface BITaxManager : NSObject {

    MCPObjectContext* _context;
    
}

/*!
    @method     initWithObjectContext:
    @abstract   initializes an instance of the class
    @discussion (comprehensive description)
    @param      context object context
    @result     returns self
*/
- (id)initWithObjectContext:(MCPObjectContext*)context;

- (MCPObjectContext*)objectContext;
- (void)setObjectContext:(MCPObjectContext*)context;


/*!
    @method     createTax
    @abstract   Creates a Tax
    @discussion Creates a Tax and creates all the necessary things needed for the tax, ie. a consolidated tax representation etc.
    @result     returns the newly created tax
*/
- (BITax*)createTax;


/*!
    @method     deleteTax:
    @abstract   Deletes a Tax
    @discussion Deletes a Tax and deletes the tax from all consolidated taxes its associated with as well as versioning new cons taxes, etc.
    @param      aTax the tax which is to be deleted
*/
- (void)deleteTax:(BITax*)aTax;


/*!
    @method     createConsolidatedTax
    @abstract   creates a consolidated tax not associated with any taxes
    @discussion (comprehensive description)
    @result     returns a consolidated tax
*/
- (BIConsolidatedTax*)createConsolidatedTax;

/*!
    @method     deleteConsolidatedTax:
    @abstract   Deletes a consolidated tax
    @discussion consolidated taxes are never deleted, they are just simply versioned/made inactive.  In this case, we are just
 making the consolidated tax inactive.  its not actually deleted.
    @param      aConsTax the consolidated tax we want to delete.
*/
- (void)deleteConsolidatedTax:(BIConsolidatedTax*)aConsTax;


/*!
    @method     addTax:toConsolidatedTax:
    @abstract   adds a tax to a consolidated tax
    @discussion (comprehensive description)
    @param      aTax the tax to add
    @param      aConsTax the consolidated tax to add the tax to
    @result     returns the (new version of the) consolidated tax
*/
- (BIConsolidatedTax*)addTax:(BITax*)aTax toConsolidatedTax:(BIConsolidatedTax*)aConsTax;

/*!
    @method     addTaxes:toConsolidatedTax:
    @abstract   adds taxes to a consolidated tax
    @discussion (comprehensive description)
    @param      taxes the taxes to add
    @param      aConsTax aConsTax the consolidated tax to add the taxes to
    @result     returns the (new version of the) consolidated tax
*/
- (BIConsolidatedTax*)addTaxes:(NSArray*)taxes toConsolidatedTax:(BIConsolidatedTax*)aConsTax;

/*!
    @method     addTax:toConsolidatedTax:createNewVersion:
    @abstract   adds a tax to a consolidated tax and only creates a new version of the consolidated tax if specified
    @discussion the main reason for this is so that we can re-use this code in situations where we have already created a new
                version of the consolidated tax and we do not need to again (ie. updating a tax)
    @param      aTax tax to add to consolidated tax
    @param      aConsTax consolidated tax to add the tax to
    @param      aFlag specifies whether a new version of the consolidated tax should be created or not
    @result     the consolidated tax (whether its the new version or the same one you passed in)
*/
- (BIConsolidatedTax*)addTax:(BITax*)aTax toConsolidatedTax:(BIConsolidatedTax*)aConsTax createNewVersion:(BOOL)aFlag;

/*!
    @method     addTaxes:toConsolidatedTax:createNewVersion:
    @abstract   adds taxes to a consolidated tax and only creates a new version of the consolidated tax if specified
    @discussion the main reason for this is so that we can re-use this code in situations where we have already created a new
                version of the consolidated tax and we do not need to again (ie. updating a tax)
    @param      taxes taxes to add to consolidated tax
    @param      aConsTax consolidated tax to add the taxes to
    @param      aFlag specifies whether a new version of the consolidated tax should be created or not
    @result     the consolidated tax (whether its the new version or the same one you passed in)
*/
- (BIConsolidatedTax*)addTaxes:(NSArray*)taxes toConsolidatedTax:(BIConsolidatedTax*)aConsTax createNewVersion:(BOOL)aFlag;

/*!
    @method     createNewVersionOfConsolidatedTax:
    @abstract   creates a new version of a consolidated tax
    @discussion disactivates old version, creates new version, copies over all taxes to the new version
    @param      aConsTax the consolidated tax which you want to get a new version of
    @result     returns a new version of the consolidated tax
*/
- (BIConsolidatedTax*)createNewVersionOfConsolidatedTax:(BIConsolidatedTax*)aConsTax;

/*!
    @method     removeTaxes:fromConsolidatedTax:
    @abstract   remove taxes from a consolidated tax
    @discussion (comprehensive description)
    @param      taxes the taxes to remove
    @param      aConsTax the consolidated tax to remove the taxes from.
    @result     returns the (new version of the) consolidated tax which no longer contains taxes
*/
- (BIConsolidatedTax*)removeTaxes:(NSArray*)taxes fromConsolidatedTax:(BIConsolidatedTax*)aConsTax;

/*!
    @method     removeTax:fromConsolidatedTax:
    @abstract   remove a tax from a consolidated tax
    @discussion (comprehensive description)
    @param      aTax tax tax to remove
    @param      aConsTax the consolidated tax to remove the tax from.
    @result     returns the (new version of the) consolidated tax which no longer contains a tax
*/
- (BIConsolidatedTax*)removeTax:(BITax*)aTax fromConsolidatedTax:(BIConsolidatedTax*)aConsTax;

/*!
    @method     updateTax:name:rate:registrationNum:compound:
    @abstract   used to update attributes of a tax
    @discussion creates a new version of the tax with the newly updated attributes (takes care of updating all consolidated taxes, etc.)
    @param      oldTax the tax whos attributes has changed
    @param      aName the modified name (if it has not been modified pass in nil)
    @param      aRate the modified rate (if it has not been modified pass in nil)
    @param      aRegNum the modified registration number (if it has not been modified pass in nil)
    @param      aCompound the modified compound flag (if it has not been modified pass in nil)
    @result     returns the newly updated tax (a new version of the tax with the updated attribute values)
*/
- (BITax*)updateTax:(BITax*)oldTax 
               name:(NSString*)aName 
               rate:(NSNumber*)aRate 
    registrationNum:(NSString*)aRegNum
           compound:(NSNumber*)aCompound;

//- (NSNumber*)taxesTotalForObject:(id)anObject consolidatedTax:(BIConsolidatedTax*)aConsTax total:(NSDecimalNumber*)aTotal;
//
//- (NSNumber*)tax:(BITax*)matchTax onTotal:(NSDecimalNumber*)aTotal object:(id)anObject;

- (void)setTax:(BITax*)orderTax index:(int)anIndex;

- (void)fixTaxOrderIndexes;

- (void)deleteAllTaxesAndConsolidatedTaxes;

- (void)restoreTaxSetup:(NSString*)aPath;

- (void)saveTaxSetupToFile:(NSString*)aFilename;

+ (NSArray*)taxesDictionarySummaryFromTaxFile:(NSString*)aFilename;
- (NSArray*)taxesDictionarySummaryFromTaxFile:(NSString*)aFilename;

- (BOOL)removeTaxFromUserDefaults:(BITax*)oldTax;
- (void)setUserDefaultsTax:(BIConsolidatedTax*)aConsTax;

@end
